#!/bin/bash

# Script : ip4calc.sh
# Author : Rene Hartman (http://hac-maarssen.nl)
# Date   : 20110203
# Purpose: Calculate netmask, prefix, broadcast, network and CIDR notation from /prefix, netmask, ipaddress/prefix or ipaddress netmask

# Remarks: Function mask2cidr is based on a linuxquestions forumpost by 'Hko'
#          Function cidr2mask is based on a linuxquestions forumpost by 'matschaffer'
#            http://www.linuxquestions.org/questions/programming-9/bash-cidr-calculator-646701/
#          Permission is granted for personal, non-commercial use.
#          The script may not be re-distributed in any form without written permission from Rene Hartman ( mail [at] hac-maarssen.nl ).
#          The script may be modified for personal use.
#          Similar functionality is available in program 'ipcalc', shipped with some Linux distributions; ipcalc also handles IPv6.
#          The output format of ip4calc.sh has been kept compatible with ipcalc's output format.  

# Caution: THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES
#            OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE AUTHOR ACCEPTS NO RESPONSIBILITY IN ANY CONCEIVABLE MANNER.

# ------------------------------ D E F A U L T S -------------------------------

# ----------------------------- F U N C T I O N S ------------------------------
# Usage
#
usage() {
  echo -e "\nip4calc.sh calculates netmask, prefix, broadcast, network and CIDR notation from\n \
          /prefix (=mask bits), netmask, ipaddress/prefix or ipaddress netmask"
  echo -e "\nUsage: $(basename $0) ipaddress/prefix\n \
        - or -\n \
      $(basename $0) ipaddress netmask\n \
        - or -\n \
      $(basename $0) /prefix\n \
        - or -\n \
      $(basename $0) netmask\n"
}

#-------------------------------------------------------------------------------
# Verify address has 4 non-empty octets
#
verify4octets() {
  ADDR=$1
  nroct=$(echo $ADDR | awk -F. '{print NF}')
  [ $nroct -ne 4 ] && echo "ERROR: invalid number of octets in $ADDR: $nroct, should be 4" && exit 8
  for ((i=1 ; i<5 ; i++)) ; do
    dec=$(echo $ADDR | cut -d. -f$i)
    [ -z "$dec" ] && echo "ERROR: octet $i in $ADDR is empty" && exit 8
  done
}

#-------------------------------------------------------------------------------
# Verify prefix
verifyprefix() {
  pfix=$1
  [ $pfix -lt 1 ] || [ $pfix -gt 32 ] && echo "ERROR: invalid value $pfix for prefix" && exit 6 
}

#-------------------------------------------------------------------------------
# Verify netmask hierarchy
#
verifynetmask() {
  ADDR=$1
  for ((i=1 ; i<5 ; i++)) ; do
    dec[$i]=$(echo $ADDR | cut -d. -f$i)
  done
  for ((i=2 ; i<5 ; i++)) ; do
     [ ${dec[$((i-1))]} -lt ${dec[$i]} ] && echo "ERROR: netmask $ADDR is invalid" && exit 2
  done
}

#-------------------------------------------------------------------------------
# Calculate number of bits in a netmask
#
mask2cidr() {
  bits=0
  IFS=.
  for dec in $1 ; do
    case $dec in
      255 ) ((bits+=8)) ;;
      254 ) ((bits+=7)) ;;
      252 ) ((bits+=6)) ;;
      248 ) ((bits+=5)) ;;
      240 ) ((bits+=4)) ;;
      224 ) ((bits+=3)) ;;
      192 ) ((bits+=2)) ;;
      128 ) ((bits+=1)) ;;
        0 )             ;;
        * ) return $dec ;;
    esac
  done
  echo "$bits"
}

#-------------------------------------------------------------------------------
# Calculate netmask from number of bits
#
cidr2mask() {
  bits=$1
  mask=""
  full_octets=$((bits/8))
  part_octet=$((bits%8))

  for ((i=0 ; i<4 ; i++)) ; do
    if [ $i -lt $full_octets ] ; then
      mask+=255
    elif [ $i -eq $full_octets ] ; then
      mask+=$((256 - 2**(8-$part_octet)))
    else
      mask+=0
    fi
    [ $i -lt 3 ] && mask+=.
  done

  echo $mask
}

#-------------------------------------------------------------------------------
# Calculate network and broadcast addresses from IP address and number of bits
#
ip2netw() {
  ipad=$1
  bits=$2
  full_octets=$((bits/8))
  part_octet=$((bits%8))

  if [ $part_octet -eq 0 ] ; then
    blksz=256
    last_octet=$full_octets
  else
    blksz=$((256/2**$part_octet))
    last_octet=$((full_octets+1))
  fi

  last_dec=$(echo $ipad | cut -d. -f$last_octet)

  for ((i=0 ; i<256 ; i+=$blksz)) ; do
    if [ $i -gt $(($last_dec-$blksz)) ] ; then
      case $full_octets in
        4 ) netw=$(echo $ipad)
            bcst=$(echo $ipad)                                        ;;
        3 ) netw=$(echo $ipad | cut -d. -f1-3).$i
            bcst=$(echo $ipad | cut -d. -f1-3).$(($i+$blksz-1))       ;;
        2 ) netw=$(echo $ipad | cut -d. -f1-2).$i.0
            bcst=$(echo $ipad | cut -d. -f1-2).$(($i+$blksz-1)).255   ;;
        1 ) netw=$(echo $ipad | cut -d. -f1).0.0.0
            bcst=$(echo $ipad | cut -d. -f1).$(($i+$blksz-1)).255.255 ;;
        0 ) netw=$i.0.0.0
            bcst=$(($i+$blksz-1)).255.255.255                         ;;
      esac
      [ -n  "$netw" ] && break
    fi
  done

  echo "$netw $bcst"
}

# ---------------------------------- M A I N -----------------------------------
# Get parameters
[ $# -eq 0 ] || [ $# -gt 2 ]  && usage && exit 9
[ $(echo $1 | grep -Fi 'h') ] && usage && exit 9

IPAP=$1
MASK=$2
IPAD=$(echo $IPAP | cut -d/ -f1)
PFIX=$(echo $IPAP | cut -d/ -f2)

# If single parameter, determine what was passed
if [ "$PFIX" == "$IPAD" ] ; then
  if [ -z "$MASK" ] ; then
    MASK=$IPAD
    IPAD=
  fi
  [ -n "$MASK" ] && verify4octets $MASK
  prefix=$(mask2cidr $MASK)
  rc=$?
  [ $rc -ne 0 ] && echo "ERROR: mask octet value $rc is invalid" && exit $rc
else
  [ -n "$MASK" ] && echo "ERROR: either prefix or netmask must be specified" && exit 9
  prefix=$PFIX
fi
[ -n "$IPAD" ] && verify4octets $IPAD
verifyprefix $prefix

if [ -z "$MASK" ] ; then
  netmask=$(cidr2mask $prefix)
else
  netmask=$MASK
fi
verifynetmask $netmask

if [ -n "$IPAD" ] ; then
  netcast=$(ip2netw $IPAD $prefix)
  bcast=$(echo $netcast | awk '{print $2}')
  network=$(echo $netcast | awk '{print $1}')
  cidrnot=$network/$prefix
fi

[ -n "$IPAD" ]    && echo "IPADDRESS=$IPAD"
[ -n "$netmask" ] && echo "NETMASK=$netmask"
[ -n "$prefix" ]  && echo "PREFIX=$prefix"
[ -n "$bcast" ]   && echo "BROADCAST=$bcast"
[ -n "$network" ] && echo "NETWORK=$network"
[ -n "$cidrnot" ] && echo "NOTATION=$network/$prefix"

exit 0